// ========================================
// إعدادات عامة
// ========================================
document.addEventListener('DOMContentLoaded', function () {

    const body = document.body;
    const filtersSidebar = document.querySelector('.filters-sidebar');
    const mobileFilterBtn = document.querySelector('.mobile-filter-btn');
    const applyFiltersBtn = document.querySelector('.apply-filters-btn');
    const clearFiltersBtn = document.querySelector('.clear-filters-btn');
    const productsGrid = document.getElementById('products-grid');
    const productsCountEl = document.getElementById('products-count');
    const noProductsEl = document.getElementById('no-products');
    const sortSelect = document.getElementById('sort');
    const activeFiltersContainer = document.getElementById('active-filters-container');
    const priceSlider = document.getElementById('price-slider');
    const minPriceInput = document.getElementById('min-price');
    const maxPriceInput = document.getElementById('max-price');
    const priceMinDisplay = document.getElementById('price-min-display');
    const priceMaxDisplay = document.getElementById('price-max-display');

    let filtersOverlay = null;

    // حفظ المنتجات الأصلية
    const allProducts = Array.from(document.querySelectorAll('.product-card'));

    // ========================================
    // فتح / إغلاق الفلاتر في الموبايل
    // ========================================
    function openFilters() {
        if (!filtersSidebar) return;
        filtersSidebar.classList.add('active');

        if (!filtersOverlay) {
            filtersOverlay = document.createElement('div');
            filtersOverlay.className = 'filters-overlay';
            document.body.appendChild(filtersOverlay);
        }
        setTimeout(() => filtersOverlay.classList.add('active'), 10);

        filtersOverlay.addEventListener('click', closeFilters);
        body.style.overflow = 'hidden';
    }

    function closeFilters() {
        if (!filtersSidebar) return;
        filtersSidebar.classList.remove('active');
        if (filtersOverlay) {
            filtersOverlay.classList.remove('active');
        }
        body.style.overflow = '';
    }

    if (mobileFilterBtn) {
        mobileFilterBtn.addEventListener('click', openFilters);
    }

    // زر X في أعلى الفلاتر (عن طريق ::before)
    if (filtersSidebar) {
        filtersSidebar.addEventListener('click', function (e) {
            if (window.innerWidth > 992) return;
            const sidebarRect = filtersSidebar.getBoundingClientRect();
            const xArea = {
                left: sidebarRect.left,
                right: sidebarRect.left + 70,
                top: sidebarRect.top,
                bottom: sidebarRect.top + 70
            };
            if (
                e.clientX >= xArea.left &&
                e.clientX <= xArea.right &&
                e.clientY >= xArea.top &&
                e.clientY <= xArea.bottom
            ) {
                closeFilters();
            }
        });
    }

    // ========================================
    // فتح/إغلاق مجموعات الفلاتر
    // ========================================
    document.querySelectorAll('.filter-toggle').forEach(toggle => {
        toggle.addEventListener('click', function () {
            const content = this.nextElementSibling;
            const icon = this.querySelector('i');
            if (!content) return;

            content.classList.toggle('active');
            if (content.classList.contains('active')) {
                content.style.maxHeight = content.scrollHeight + 'px';
                icon.style.transform = 'rotate(180deg)';
            } else {
                content.style.maxHeight = '0px';
                icon.style.transform = 'rotate(0deg)';
            }
        });

        // ضبط المجموعات المفتوحة افتراضياً
        const content = toggle.nextElementSibling;
        if (content && content.classList.contains('active')) {
            content.style.maxHeight = content.scrollHeight + 'px';
        }
    });

    // ========================================
    // تحديث عرض السعر
    // ========================================
    function syncPriceUI() {
        const minVal = parseInt(minPriceInput.value) || 0;
        const maxVal = parseInt(maxPriceInput.value) || 0;
        priceMinDisplay.textContent = `${minVal} جنيه`;
        priceMaxDisplay.textContent = `${maxVal.toLocaleString()} جنيه`;
        priceSlider.value = maxVal;
    }

    if (priceSlider && minPriceInput && maxPriceInput) {
        priceSlider.addEventListener('input', function () {
            maxPriceInput.value = this.value;
            syncPriceUI();
        });

        minPriceInput.addEventListener('change', () => {
            if (+minPriceInput.value < 0) minPriceInput.value = 0;
            if (+minPriceInput.value > +maxPriceInput.value)
                minPriceInput.value = maxPriceInput.value;
            syncPriceUI();
        });

        maxPriceInput.addEventListener('change', () => {
            if (+maxPriceInput.value < +minPriceInput.value)
                maxPriceInput.value = minPriceInput.value;
            syncPriceUI();
        });

        syncPriceUI();
    }

    // ========================================
    // إنشاء / تحديث Tags الفلاتر النشطة
    // ========================================
    function updateActiveFilterTags() {
        activeFiltersContainer.innerHTML = '';

        const selectedCheckboxes = document.querySelectorAll(
            '.filters-sidebar input[type="checkbox"]:checked'
        );

        if (selectedCheckboxes.length === 0 &&
            (+minPriceInput.value === 0 && +maxPriceInput.value === 10000)) {
            return;
        }

        const labelEl = document.createElement('span');
        labelEl.className = 'filter-label';
        labelEl.textContent = 'الفلاتر النشطة:';
        activeFiltersContainer.appendChild(labelEl);

        const tagsWrapper = document.createElement('div');
        tagsWrapper.className = 'filter-tags';
        activeFiltersContainer.appendChild(tagsWrapper);

        selectedCheckboxes.forEach(input => {
            const parentLabel = input.closest('.filter-option');
            if (!parentLabel) return;
            const text = parentLabel.querySelector('.option-text')?.textContent || '';
            const tag = document.createElement('span');
            tag.className = 'filter-tag';
            tag.dataset.filterName = input.name;
            tag.dataset.filterValue = input.value;
            tag.innerHTML = `${text}<i class="fas fa-times"></i>`;
            tagsWrapper.appendChild(tag);
        });

        // Tag للسعر لو متغير
        if (+minPriceInput.value !== 0 || +maxPriceInput.value !== 10000) {
            const tag = document.createElement('span');
            tag.className = 'filter-tag';
            tag.dataset.filterName = 'price';
            tag.innerHTML =
                `السعر: ${minPriceInput.value} - ${Number(maxPriceInput.value).toLocaleString()} جنيه` +
                `<i class="fas fa-times"></i>`;
            tagsWrapper.appendChild(tag);
        }

        // إزالة الفلتر من الـ tag
        tagsWrapper.querySelectorAll('.filter-tag i').forEach(icon => {
            icon.addEventListener('click', function () {
                const tag = this.parentElement;
                const name = tag.dataset.filterName;
                const value = tag.dataset.filterValue;

                if (name === 'price') {
                    minPriceInput.value = 0;
                    maxPriceInput.value = 10000;
                    syncPriceUI();
                } else {
                    const cb = document.querySelector(
                        `.filters-sidebar input[name="${name}"][value="${value}"]`
                    );
                    if (cb) cb.checked = false;
                }

                filterProducts();
            });
        });
    }

    // ========================================
    // فلترة المنتجات فعلياً
    // ========================================
    function filterProducts() {
        const selectedCities = Array.from(
            document.querySelectorAll('input[name="city"]:checked')
        ).map(el => el.value);

        const selectedMain = Array.from(
            document.querySelectorAll('input[name="main-category"]:checked')
        ).map(el => el.value);

        const selectedSub = Array.from(
            document.querySelectorAll('input[name="sub-category"]:checked')
        ).map(el => el.value);

        const selectedColors = Array.from(
            document.querySelectorAll('input[name="color"]:checked')
        ).map(el => el.value);

        const selectedSizes = Array.from(
            document.querySelectorAll('input[name="size"]:checked')
        ).map(el => el.value);

        const minPrice = parseInt(minPriceInput.value) || 0;
        const maxPrice = parseInt(maxPriceInput.value) || 10000;

        let visibleCount = 0;

        allProducts.forEach(card => {
            const city = card.dataset.city;
            const mainCat = card.dataset.mainCategory;
            const subCat = card.dataset.subCategory;
            const color = card.dataset.color;
            const size = card.dataset.size;
            const price = parseInt(card.dataset.price) || 0;

            let show = true;

            if (selectedCities.length && !selectedCities.includes(city)) show = false;
            if (selectedMain.length && !selectedMain.includes(mainCat)) show = false;
            if (selectedSub.length && !selectedSub.includes(subCat)) show = false;
            if (selectedColors.length && !selectedColors.includes(color)) show = false;
            if (selectedSizes.length && !selectedSizes.includes(size)) show = false;
            if (price < minPrice || price > maxPrice) show = false;

            if (show) {
                card.style.display = '';
                visibleCount++;
            } else {
                card.style.display = 'none';
            }
        });

        productsCountEl.textContent = visibleCount;

        if (visibleCount === 0) {
            noProductsEl.style.display = 'block';
        } else {
            noProductsEl.style.display = 'none';
        }

        updateActiveFilterTags();

        // إغلاق الفلاتر في الموبايل بعد التطبيق
        if (window.innerWidth <= 992) {
            closeFilters();
        }
    }

    if (applyFiltersBtn) {
        applyFiltersBtn.addEventListener('click', filterProducts);
    }

    // كل checkbox يفلتر مباشرة لو حابب تشتغل Live
    document.querySelectorAll('.filters-sidebar input[type="checkbox"]').forEach(cb => {
        cb.addEventListener('change', filterProducts);
    });

    // ========================================
    // مسح كل الفلاتر
    // ========================================
    if (clearFiltersBtn) {
        clearFiltersBtn.addEventListener('click', () => {
            document
                .querySelectorAll('.filters-sidebar input[type="checkbox"]')
                .forEach(cb => (cb.checked = false));

            minPriceInput.value = 0;
            maxPriceInput.value = 10000;
            syncPriceUI();

            filterProducts();
        });
    }

    // ========================================
    // ترتيب المنتجات
    // ========================================
    function sortProducts() {
        const value = sortSelect.value;
        const cards = allProducts.slice(); // نسخة

        if (value === 'price-asc') {
            cards.sort((a, b) => a.dataset.price - b.dataset.price);
        } else if (value === 'price-desc') {
            cards.sort((a, b) => b.dataset.price - a.dataset.price);
        } else if (value === 'name-asc') {
            cards.sort((a, b) =>
                a.querySelector('.product-title').textContent
                    .localeCompare(b.querySelector('.product-title').textContent, 'ar')
            );
        } else if (value === 'name-desc') {
            cards.sort((a, b) =>
                b.querySelector('.product-title').textContent
                    .localeCompare(a.querySelector('.product-title').textContent, 'ar')
            );
        }
        // الأحدث / الأكثر مبيعاً ممكن تعتمد على data attributes إضافية

        cards.forEach(card => productsGrid.appendChild(card));
    }

    if (sortSelect) {
        sortSelect.addEventListener('change', sortProducts);
    }

    // ========================================
    // مفضلة المنتجات
    // ========================================
    document.querySelectorAll('.product-wishlist').forEach(btn => {
        btn.addEventListener('click', function (e) {
            e.preventDefault();
            this.classList.toggle('active');
            const icon = this.querySelector('i');
            if (this.classList.contains('active')) {
                icon.classList.remove('far');
                icon.classList.add('fas');
            } else {
                icon.classList.remove('fas');
                icon.classList.add('far');
            }
        });
    });

    // ========================================
    // تبديل Grid / List
    // ========================================
    const viewBtns = document.querySelectorAll('.view-btn');

    viewBtns.forEach(btn => {
        btn.addEventListener('click', function () {
            viewBtns.forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            const view = this.dataset.view;

            if (view === 'list') {
                productsGrid.style.gridTemplateColumns = '1fr';
                allProducts.forEach(card => {
                    card.style.display = card.style.display === 'none' ? 'none' : 'grid';
                    card.style.gridTemplateColumns = '260px 1fr';
                });
            } else {
                productsGrid.style.gridTemplateColumns =
                    'repeat(auto-fill, minmax(260px, 1fr))';
                allProducts.forEach(card => {
                    card.style.display = card.style.display === 'none' ? 'none' : 'block';
                    card.style.gridTemplateColumns = '';
                });
            }
        });
    });

    // ========================================
    // بحث المحافظات البسيط
    // ========================================
    const citySearch = document.querySelector('.city-search');
    if (citySearch) {
        citySearch.addEventListener('input', function () {
            const term = this.value.trim().toLowerCase();
            document.querySelectorAll('.city-options .filter-option').forEach(opt => {
                const text = opt
                    .querySelector('.option-text')
                    .textContent.toLowerCase();
                opt.style.display = text.includes(term) ? 'flex' : 'none';
            });
        });
    }

    // ========================================
    // Pop-up المعاينة السريعة
    // ========================================
    const quickViewModal = document.getElementById('quickViewModal');
    const modalOverlay = quickViewModal.querySelector('.modal-overlay');
    const modalClose = quickViewModal.querySelector('.modal-close');
    const modalImage = document.getElementById('modalImage');
    const modalCategory = document.getElementById('modalCategory');
    const modalTitle = document.getElementById('modalTitle');
    const modalRating = document.getElementById('modalRating');
    const modalPrice = document.getElementById('modalPrice');
    const modalDescription = document.getElementById('modalDescription');
    const modalColors = document.getElementById('modalColors');
    const modalSizes = document.getElementById('modalSizes');
    const modalViewMore = document.getElementById('modalViewMore');
    const modalWishlistBtn = quickViewModal.querySelector('.modal-btn-wishlist');

    function openQuickView(productData, productId) {
        if (!productData) return;

        modalImage.src = productData.image;
        modalImage.alt = productData.title;
        modalCategory.textContent = productData.category;
        modalTitle.textContent = productData.title;

        // Rating
        modalRating.innerHTML = '';
        const ratingVal = parseFloat(productData.rating || '0');
        const full = Math.floor(ratingVal);
        const half = ratingVal - full >= 0.5;
        for (let i = 0; i < 5; i++) {
            const icon = document.createElement('i');
            if (i < full) icon.className = 'fas fa-star';
            else if (i === full && half) icon.className = 'fas fa-star-half-alt';
            else icon.className = 'far fa-star';
            modalRating.appendChild(icon);
        }
        const span = document.createElement('span');
        span.textContent = `(${productData.reviews || 0})`;
        modalRating.appendChild(span);

        // Price
        modalPrice.innerHTML = '';
        const curr = document.createElement('span');
        curr.className = 'current-price';
        curr.textContent = `${productData.price} جنيه`;
        modalPrice.appendChild(curr);
        if (productData.oldPrice) {
            const old = document.createElement('span');
            old.className = 'old-price';
            old.textContent = `${productData.oldPrice} جنيه`;
            modalPrice.appendChild(old);
        }

        modalDescription.textContent = productData.description || '';

        // Colors
        modalColors.innerHTML = '';
        (productData.colors || []).forEach(c => {
            const dot = document.createElement('span');
            dot.className = 'color-dot';
            dot.style.background = c;
            if (c === '#fff') dot.style.border = '1px solid #ddd';
            modalColors.appendChild(dot);
        });

        // Sizes
        modalSizes.innerHTML = '';
        (productData.sizes || []).forEach(s => {
            const box = document.createElement('span');
            box.className = 'size-box';
            box.textContent = s;
            modalSizes.appendChild(box);
        });

        modalViewMore.href = `product-details.html?id=${productId}`;

        quickViewModal.classList.add('active');
        body.style.overflow = 'hidden';
    }

    function closeQuickView() {
        quickViewModal.classList.remove('active');
        body.style.overflow = '';
    }

    document.querySelectorAll('.quick-view-btn').forEach((btn, index) => {
        btn.addEventListener('click', function (e) {
            e.preventDefault();
            const dataStr = this.getAttribute('data-product');
            try {
                const data = JSON.parse(dataStr);
                const card = this.closest('.product-card');
                const idFromHref =
                    card
                        .querySelector('.view-details-btn')
                        ?.getAttribute('href')
                        ?.match(/id=(\d+)/)?.[1] || index + 1;
                openQuickView(data, idFromHref);
            } catch (err) {
                console.error('Invalid product data', err);
            }
        });
    });

    modalOverlay.addEventListener('click', closeQuickView);
    modalClose.addEventListener('click', closeQuickView);
    document.addEventListener('keyup', e => {
        if (e.key === 'Escape') closeQuickView();
    });

    modalWishlistBtn.addEventListener('click', function () {
        this.classList.toggle('active');
        const icon = this.querySelector('i');
        if (this.classList.contains('active')) {
            icon.classList.remove('far');
            icon.classList.add('fas');
        } else {
            icon.classList.remove('fas');
            icon.classList.add('far');
        }
    });

    // تشغيل أول فلترة لتحديث العدد والـ tags
    filterProducts();
});
