// ========================================
// صفحة اتصل بنا
// ========================================

document.addEventListener('DOMContentLoaded', function() {

    // ========================================
    // معالجة نموذج الاتصال
    // ========================================
    const contactForm = document.getElementById('contactForm');
    const submitBtn = contactForm.querySelector('.submit-btn');
    const btnText = submitBtn.querySelector('.btn-text');
    const btnLoader = submitBtn.querySelector('.btn-loader');
    const successMessage = contactForm.querySelector('.success-message');

    if (contactForm) {
        contactForm.addEventListener('submit', function(e) {
            e.preventDefault();

            // التحقق من صحة البيانات
            if (!validateForm()) {
                return;
            }

            // تعطيل الزر وإظهار Loader
            submitBtn.disabled = true;
            btnText.style.display = 'none';
            btnLoader.style.display = 'inline-block';

            // محاكاة إرسال البيانات
            const formData = new FormData(contactForm);
            const data = Object.fromEntries(formData);

            console.log('Form Data:', data);

            // محاكاة استجابة الخادم
            setTimeout(() => {
                // إخفاء Loader
                btnLoader.style.display = 'none';
                btnText.style.display = 'inline';
                submitBtn.disabled = false;

                // إظهار رسالة النجاح
                successMessage.style.display = 'flex';

                // إعادة تعيين النموذج
                contactForm.reset();
                updateCharCount();
                clearFileUpload();

                // إخفاء رسالة النجاح بعد 5 ثواني
                setTimeout(() => {
                    successMessage.style.display = 'none';
                }, 5000);

            }, 2000);
        });
    }

    // ========================================
    // التحقق من صحة النموذج
    // ========================================
    function validateForm() {
        let isValid = true;

        // التحقق من الاسم
        const name = document.getElementById('name');
        if (!name.value.trim()) {
            showError(name, 'الاسم مطلوب');
            isValid = false;
        } else {
            removeError(name);
        }

        // التحقق من البريد الإلكتروني
        const email = document.getElementById('email');
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!email.value.trim()) {
            showError(email, 'البريد الإلكتروني مطلوب');
            isValid = false;
        } else if (!emailRegex.test(email.value)) {
            showError(email, 'البريد الإلكتروني غير صحيح');
            isValid = false;
        } else {
            removeError(email);
        }

        // التحقق من رقم الهاتف
        const phone = document.getElementById('phone');
        if (!phone.value.trim()) {
            showError(phone, 'رقم الهاتف مطلوب');
            isValid = false;
        } else if (phone.value.replace(/\s/g, '').length < 10) {
            showError(phone, 'رقم الهاتف غير صحيح');
            isValid = false;
        } else {
            removeError(phone);
        }

        // التحقق من الموضوع
        const subject = document.getElementById('subject');
        if (!subject.value) {
            showError(subject, 'الموضوع مطلوب');
            isValid = false;
        } else {
            removeError(subject);
        }

        // التحقق من الرسالة
        const message = document.getElementById('message');
        if (!message.value.trim()) {
            showError(message, 'الرسالة مطلوبة');
            isValid = false;
        } else if (message.value.length < 10) {
            showError(message, 'الرسالة يجب أن تكون 10 أحرف على الأقل');
            isValid = false;
        } else {
            removeError(message);
        }

        // التحقق من الموافقة
        const agree = document.getElementById('agree');
        if (!agree.checked) {
            alert('يجب الموافقة على سياسة الخصوصية');
            isValid = false;
        }

        return isValid;
    }

    function showError(element, message) {
        const formGroup = element.closest('.form-group');
        formGroup.classList.add('error');
        const errorMessage = formGroup.querySelector('.error-message');
        if (errorMessage) {
            errorMessage.textContent = message;
        }
        element.classList.add('error');
    }

    function removeError(element) {
        const formGroup = element.closest('.form-group');
        formGroup.classList.remove('error');
        element.classList.remove('error');
    }

    // ========================================
    // التحقق من البريد الإلكتروني مباشرة
    // ========================================
    const emailInput = document.getElementById('email');
    if (emailInput) {
        emailInput.addEventListener('blur', function() {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (this.value && !emailRegex.test(this.value)) {
                showError(this, 'البريد الإلكتروني غير صحيح');
            } else if (this.value) {
                removeError(this);
            }
        });
    }

    // ========================================
    // عداد الأحرف للرسالة
    // ========================================
    const messageTextarea = document.getElementById('message');
    const charCount = document.querySelector('.char-count');

    function updateCharCount() {
        if (messageTextarea && charCount) {
            const length = messageTextarea.value.length;
            charCount.textContent = `${length} / 500`;
            
            if (length > 500) {
                charCount.style.color = '#dc2626';
            } else if (length > 400) {
                charCount.style.color = '#f59e0b';
            } else {
                charCount.style.color = '#999';
            }
        }
    }

    if (messageTextarea) {
        messageTextarea.addEventListener('input', updateCharCount);
        
        // منع إدخال أكثر من 500 حرف
        messageTextarea.addEventListener('keydown', function(e) {
            if (this.value.length >= 500 && e.key !== 'Backspace' && e.key !== 'Delete') {
                e.preventDefault();
            }
        });
    }

    // ========================================
    // رفع الملف
    // ========================================
    const fileInput = document.getElementById('attachment');
    const fileLabel = document.querySelector('.file-label');
    const fileText = fileLabel.querySelector('.file-text');
    const fileName = fileLabel.querySelector('.file-name');

    if (fileInput) {
        fileInput.addEventListener('change', function() {
            if (this.files && this.files[0]) {
                const file = this.files[0];
                const fileSizeMB = file.size / 1024 / 1024;

                if (fileSizeMB > 5) {
                    alert('حجم الملف يجب أن يكون أقل من 5 ميجابايت');
                    this.value = '';
                    clearFileUpload();
                    return;
                }

                fileText.style.display = 'none';
                fileName.textContent = file.name;
                fileName.style.display = 'block';
                fileLabel.style.borderColor = '#15803d';
                fileLabel.style.background = '#f0fdf4';
            }
        });
    }

    function clearFileUpload() {
        if (fileInput) {
            fileInput.value = '';
            fileText.style.display = 'block';
            fileName.style.display = 'none';
            fileName.textContent = '';
            fileLabel.style.borderColor = '#e0e0e0';
            fileLabel.style.background = '#f9f9f9';
        }
    }

    // ========================================
    // FAQ Accordion
    // ========================================
    const faqQuestions = document.querySelectorAll('.faq-question');

    faqQuestions.forEach(question => {
        question.addEventListener('click', function() {
            const faqItem = this.parentElement;
            const isActive = faqItem.classList.contains('active');

            // إغلاق جميع الأسئلة
            document.querySelectorAll('.faq-item').forEach(item => {
                item.classList.remove('active');
            });

            // فتح السؤال الحالي إذا لم يكن مفتوحاً
            if (!isActive) {
                faqItem.classList.add('active');
            }
        });
    });

    // ========================================
    // Smooth Scroll للروابط
    // ========================================
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            const href = this.getAttribute('href');
            if (href === '#' || href === 'javascript:void(0)') return;
            
            e.preventDefault();
            const target = document.querySelector(href);
            if (target) {
                const offsetTop = target.getBoundingClientRect().top + window.pageYOffset - 100;
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    });

    // ========================================
    // إزالة الأخطاء عند الكتابة
    // ========================================
    const formInputs = document.querySelectorAll('.contact-form input, .contact-form select, .contact-form textarea');
    
    formInputs.forEach(input => {
        input.addEventListener('input', function() {
            if (this.value.trim()) {
                removeError(this);
            }
        });
    });

    // ========================================
    // تنسيق رقم الهاتف تلقائياً
    // ========================================
    const phoneInput = document.getElementById('phone');
    
    if (phoneInput) {
        phoneInput.addEventListener('input', function(e) {
            // السماح فقط بالأرقام والرموز + والمسافات
            this.value = this.value.replace(/[^0-9+\s]/g, '');
        });
    }

    // ========================================
    // Animation عند Scroll
    // ========================================
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -100px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);

    // تطبيق الأنيميشن على العناصر
    const animatedElements = document.querySelectorAll('.info-card, .faq-item');
    
    animatedElements.forEach((element, index) => {
        element.style.opacity = '0';
        element.style.transform = 'translateY(30px)';
        element.style.transition = `all 0.6s ease ${index * 0.1}s`;
        observer.observe(element);
    });

    // ========================================
    // تحميل الصفحة
    // ========================================
    window.addEventListener('load', function() {
        // تشغيل الأنيميشن للعناصر المرئية
        animatedElements.forEach((element, index) => {
            const rect = element.getBoundingClientRect();
            if (rect.top < window.innerHeight) {
                setTimeout(() => {
                    element.style.opacity = '1';
                    element.style.transform = 'translateY(0)';
                }, index * 100);
            }
        });
    });

});
